;+
;NAME:
;rescale_3d_fff
;PURPOSE:
;Rescales a 3d array, not by interpolating, but by summing volume
;elements, and taking the average value.
;CALLING SEQUENCE:
; f_new = rescale_3d_fff(f, scale_factor, x_new, y_new, z_new, $
;                    x_in=x_in, y_in=y_in, z_in=z_in)
;INPUT:
; f = an array
; scale_factor = the scale that you want to reduce by,n_elements(x,y
;                and z) must be integer multiples of this.
;OUTPUT:
; f_new = array rescaled down to original size by a factor of
;         scale_factor, but with values on the edges 
; x_new = the x coordinates for the values of f
; y_new = the y coordinates for the values of f
;KEYWORDS;
; x_in = an input coordinate system, the default is findgen(n_elements(y))
Function rescale_3d_fff, f, scale_factor, x_new, y_new, z_new, $
                         x_in = x_in, y_in = y_in, z_in = z_in, $
                         _extra = _extra

  sc = long(scale_factor)
  nx = n_elements(f[*, 0, 0])
  ny = n_elements(f[0, *, 0])
  nz = n_elements(f[0, 0, *])
  If(keyword_set(x_in)) Then Begin
    If(n_elements(x_in) Eq nx) Then x = x_in Else x = findgen(nx)
  Endif Else x = findgen(nx)
  If(keyword_set(y_in)) Then Begin
    If(n_elements(y_in) Eq ny) Then y = y_in Else y = findgen(ny)
  Endif Else y = findgen(ny)
  If(keyword_set(z_in)) Then Begin
    If(n_elements(z_in) Eq nz) Then z = z_in Else z = findgen(nz)
  Endif Else z = findgen(nz)

  f_new = f
  y_new = y & x_new = x & z_new = z
  If(nx mod sc Ne 0 Or ny mod sc Ne 0 Or nz Mod sc Ne 0) Then Begin
    message, /info, 'No Rescaling done'
    Return, f_new
  Endif

  n0 = nx/sc & n1 = ny/sc & n2 = nz/sc
  f1 = rebin(f, n0, n1, n2)
  x1 = rebin(x, n0)
  y1 = rebin(y, n1)
  z1 = rebin(z, n2)

  n0new = n0+1
  n1new = n1+1
  n2new = n2+1

  x_new = x[0]+(x[nx-1]-x[0])*findgen(n0new)/float(n0)
  y_new = y[0]+(y[ny-1]-y[0])*findgen(n1new)/float(n1)
  z_new = z[0]+(z[nz-1]-z[0])*findgen(n2new)/float(n2)

  x_old = [x[0], x1, x[nx-1]]
  y_old = [y[0], y1, y[ny-1]]
  z_old = [z[0], z1, z[nz-1]]
  nx_old = n_elements(x_old)
  ny_old = n_elements(y_old)
  nz_old = n_elements(z_old)
  f_old = fltarr(nx_old, ny_old, nz_old)

  ftmp = f[*, *, 0]
  ftmp = rescale_2d_fff(ftmp, sc, xd, yd, x_in = x, y_in = y, /no_interp)
  f_old[*, *, 0] = ftmp
  ftmp = f[*, *, nz-1]
  ftmp = rescale_2d_fff(ftmp, sc, xd, yd, x_in = x, y_in = y, /no_interp)
  f_old[*, *, nz_old-1] = ftmp

  ftmp = reform(f[*, 0, *])
  ftmp = rescale_2d_fff(ftmp, sc, xd, zd, x_in = x, y_in = z, /no_interp)
  f_old[*, 0, *] = ftmp

  ftmp = reform(f[*, ny-1, *])
  ftmp = rescale_2d_fff(ftmp, sc, xd, zd, x_in = x, y_in = z, /no_interp)
  f_old[*, ny_old-1, *] = ftmp

  ftmp = reform(f[0, *, *])
  ftmp = rescale_2d_fff(ftmp, sc, yd, zd, x_in = y, y_in = z, /no_interp)
  f_old[0, *, *] = ftmp

  ftmp = reform(f[nx-1, *, *])
  ftmp = rescale_2d_fff(ftmp,sc, yd, zd, x_in = y, y_in = z, /no_interp)
  f_old[nx_old-1, *, *] = ftmp

  f_old[1:nx_old-2, 1:ny_old-2, 1:nz_old-2] = f1

  f_new = temp_interp_xyz(f_old, x_old, y_old, z_old, x_new, y_new, z_new, $
                         _extra = _extra)
  Return, f_new
End
