;+
;NAME:
; rescale_2d_fff
;PURPOSE:
; rescales an array of NxM to
; (N/(scale_factor)+1)x(M/(scale_factor)+1) by rebinning, and
; then interpolating the rebinned data back to an array that
; explicitly includes the end points.
;CALLING SEQUENCE:
; f_new = rescale_2d_fff(f, scale_factor, x_new, y_new, x_in=x_in, y_in=y_in)
;INPUT:
; f = an array
; scale_factor = the scale that you want to reduce by,n_elements(x)
;                must be an integer multiple of this.
;OUTPUT:
; f_new = an array of (n_elements(f[*,0])/scale_factor)+1 by
;         (n_elements(f[0,*])/scale_factor)+1 with the same values
;         at the end points as f
; x_new = the x coordinates for the values of f
; y_new = the y coordinates for the values of f
;KEYWORDS;
; x_in = an input coordinate system, the default is findgen(n_elements(x))
; y_in = an input coordinate system, the default is
; findgen(n_elements(y))
; no_interp = if set, return the un-interpolated coordinates and
;             array, this is for use in rescale_3d.pro
;HISTORY:
; 11-jan-2008, jmm, jimm@ssl.berkeley.edu
;-
Function rescale_2d_fff, f, scale_factor, x_new, y_new, $
                         x_in = x_in, y_in = y_in, $
                         no_interp = no_interp, _extra = _extra

  sc = long(scale_factor)
  nx = n_elements(f[*, 0])
  ny = n_elements(f[0, *])
  If(keyword_set(x_in)) Then Begin
    If(n_elements(x_in) Eq nx) Then x = x_in Else x = findgen(nx)
  Endif Else x = findgen(nx)
  If(keyword_set(y_in)) Then Begin
    If(n_elements(y_in) Eq ny) Then y = y_in Else y = findgen(ny)
  Endif Else y = findgen(ny)

  f_new = f
  y_new = y & x_new = x
  If(nx mod sc Ne 0 Or ny mod sc Ne 0) Then Begin
    message, /info, 'No Rescaling done'
    Return, f_new
  Endif

  n0 = nx/sc
  n1 = ny/sc
  f1 = rebin(f, n0, n1)
  x1 = rebin(x, n0)
  y1 = rebin(y, n1)
  
  n0new = n0+1
  n1new = n1+1

  x_new = x[0]+(x[nx-1]-x[0])*findgen(n0new)/float(n0)
  y_new = y[0]+(y[ny-1]-y[0])*findgen(n1new)/float(n1)

  x_old = [x[0], x1, x[nx-1]]
  y_old = [y[0], y1, y[ny-1]]
  nx_old = n_elements(x_old)
  ny_old = n_elements(y_old)
  f_old = fltarr(nx_old, ny_old)
  f_old(1:nx_old-2, 1:ny_old-2) = f1

  f_old[*, 0] = [f[0, 0], rebin(f[*, 0], n0), f[nx-1, 0]]
  f_old[*, ny_old-1] = [f[0, ny-1], rebin(f[*, ny-1], n0), f[nx-1, ny-1]]

  f_old[0, *] = transpose([f[0, 0], rebin(transpose(f[0, *]), n1), f[0, ny-1]])
  f_old[nx_old-1, *] = transpose([f[nx-1, 0], rebin(transpose(f[nx-1, *]), n1), f[nx-1, ny-1]])

  If(keyword_set(no_interp)) Then Begin
    x_new = x_old
    y_new = y_old
    f_new = f_old
  Endif Else f_new = interpu_2d(f_old, x_old, y_old, x_new, y_new)
  Return, f_new
End

  
    
