;+
;NAME:
;   optimization_fff
;PURPOSE:
;   Program to extrapolate linear force-free fields, based on the
;   optimization algorithm published in Apj 540, 1150.
;   For more documentation, and some examples see:
;   http://sprg.ssl.berkeley.edu/fff/optimization_fff.html
;CALLING SEQUENCE:
; Optimization_fff, bx_in, by_in, bz_in, $
;           x_in = x_in, y_in = y_in, z_in = z_in, $
;           filename = filename, Restart_flag = restart_flag, $ 
;           Bc_flag = bc_flag, $
;           potlin = potlin, iterations = iterations, $
;           Dt = dt, nsave = nsave, double = double, $
;           convergence_level = convergence_level, $
;           pr_iter = pr_iter, $
;           use_fortran = use_fortran, $
;           fortran_exe = fortran_exe, $
;           use_sdf = use_sdf
;INPUT:
; bx_in, by_in, bz_in = the initial Bfield on the lower boundary, for
;                       bc_flag=0, the initial bfield for the whole
;                       volume, for bc_flag=1. These need not be
;                       passed in, if there is an input filename set.
;                       Be sure that the number of x and y points are
;                       the same. Z doesn't have to be
;OUTPUT:
; None explicit: the finished product ends up in a file called
; 'bfield.dat', which can be red in using the routine READ_BFIELD_FFF.
; Other output files are 'lhist.dat' which details the iteration
; history, and 'pfield.dat' which contains the actual initial field used.
;KEYWORDS:
; x_in, y_in, z_in = the input grids to be used. The default is to
;                    evenly divide x and y between 1 and -1, given the
;                    number of points in the input B's, and have z go
;                    from 0 to 1 with the same spatial resolution (so
;                    that if tsize=n_elements(x) then vsize =
;                    n_elements(z) is tsize/2. X and Y grids should
;                    have the same number of elements.
; restart_flag = if set, then you are restarting an old calulation,
;                and it isn't necessary to redo the initial
;                conditions. The full field must be passed in or read
;                in from a file for this case. Default is 0
; bc_flag = 0 if only the lower boundary conditions are used (typical
;             for "real data") bc_flag=1 if all the boundaries are to
;             be used. If this flag is set, be sure that B for the
;             whole volume is passed in, or read in.. Default is 0
; bz_only = if set, then when creating the initial field, only Bz at
;           the lower boundary is set to the input values, By and Bx
;           on the boundary are then the same as whatever the initial
;           potential or linear FFF give on the lower boundary; they
;           are not set to be equal to Bx_in and By_in. No effect if
;           bc_flag is set to 1.. Default is 0
; potlin = If 0 or unset, then the initial field is a potential
;          extrapolation (from Tom Metcalfs FFF.pro) If set to 1 then
;          the initial field is set to a linear FFF extrapolation.(If
;          set to 2 to the the fiels is linear FFF near the origin, but
;          varies gradually to be potential on the outer
;          boundaries. Shouldn't give a much different answer than
;          potlin=0). . Default is 0
; iterations = The max number of iterations allowed. The default is
;              10000
; dt = the step size in t. The default is 1.0e-5
; nsave = all of the output is save every nsave iterations, the
;         default is 500.
; double = if set, use double precision, this is recommended
; convergence_level = If the fractional difference objective function
;                     L is less than this then the program stops.
; pr_iter = Every pr_iter iterations, an entry to the 'lhist.dat' file is
;           made, i.e.,  
;         printf, lhist_unit, i, dt, lv, 
;           The columns in this file are (1) the iteration, (2) the value of
;           dt, (3) the current value of L the objective function
; filename = an input filename, it looks as if it depends on what the
;            boundary conditions are. B for the whole volume is
;            included:(as you would write it out)
;
;            printf, unit, rsize, tsize, vsize ;rsize=n_elements(x), tsize=n_elements(y),vsize=n_elements(z)
;            printf, unit, bx
;            printf, unit, by
;            printf, unit, bz
;            printf, unit, x
;            printf, unit, y
;            printf, unit, z
; It's easier to just pass in the fields as arrays, but this can be
; useful with the restart flag
; use_fortran = if set, this will do the iterations useing the new
;               fortran program
; fortran_exe = the name of the fortran routine used, the default is
;               'optimization_fff.exe' 
; use_sdf = if set, all data i/o is dealt with using SDF ('simple data
;           format') files. This includes the stuff that is passed to
;           the FORTRAN program. THese files will have .sdf as a suffix
;HISTORY:
;/* main.c 
; *
; * 3D force-free fields code, implementing George's optimization approach
; * This version is designed to test the method against known solutions,
; * with all of the boundary data provided (bc_flag==1) or with just the
; * lower boundary values given (bc_flag==0) and the side and top
; * boundaries fixed at the starting values (potential field values).
; * The code accepts nonuniform (Cartesian) grids. 
; *
; * 03282000: (jmm) Added the option to set the field on the boundary to 0
; * by setting set_b0 = 1 in input.dat
; * 122099: (jmm) Added a weighting term proportional to B^2, this is set
; * to 1.0 outside a shell The field should eveolve to 0.0 outside the shell
; *
; * VERSION 110799: 1/B^2 weighting introduced, to see if weak field
; * regions behave better, following Peter's idea. The modification is
; * to in the omega functions, in the evaluation of the integral L, and
; * in the addition of a term to the time evolution of the magnetic field
; * (see notes). Also added additional diagnostics whilst it is
; * calculating, that are written to lhist.dat.
; *
; * 071899: added linear force-free starting point
; *
; * 072299: made it possible to restart where the code left off and rewrote
; * to obtain input from a file (input.dat) rather than the keyboard, so
; * the code can be put in background mode to begin with. The format of 
; * the input file is specified in the file
; * POTLIN (only used if restart == 0)
; * 
; */
;IDL version feb-2001, jmm
; Added bx, by,bz, x, y, z inputs
; Added taper of initial field, 20-aug-2001, jmm
; Added potlin=2 option, the initial state is linear FFF at R=0, goes
; to potential at the boundaries, in principle, this should give the
; same result as the potential field initial condition, since the
; boundaries are the same... jmm, 19-dec-2003. 
; Changed from using calc_lin_fff to Tom Metcalf's fff.pro, its faster
; and better, 8-jan-2004, jmm
; Added bz_only option, for those cases without vector
; magnetograms. jmm, 21-jan-2004
; Trying for more speed, added a common block for curl, grad and div
; of b, the idea is to only update once per iteration, instead of 3
; times, jmm 18-feb-2004
; Added gray_middle option, in which the field inside the boundaries
; is initialized to a constant value, just to see if it works out.
; passed through to initialb_fff, jmm, 30-apr-2004
; Re-added option to use calc_lin_fff, the /slow_way keyword, added
; alpha_in keyword, 26-may-2004, jmm
; Added reset_dt, at every j mod reset_dt, dt is reset to the original
; value, may be useful? If reset_dt is not set, no resetting of dt.
; 29-dec-2004, jmm, added spherical keyword, for spherical coordinates
; x stands for r, y stands for theta, z stands for phi
; 13-jan-2005, jmm, allow for number of x to not equal number of y
; points, all sorts of other changes to allow spherical coordinates
; Inputs are now required to be all 3 d arrays. Now rsize refers to
; the first coordinate, tsize to the second one and vsize to the third
; one.
; Added use_fortran, and fortran_exe keywords, 28-may-2006, jmm
; Added multiplicative weighting function, 6-jul-2007, jmm
;-
Pro Optimization_fff, bx_in, by_in, bz_in, $
                      Restart_flag = restart_flag, $ 
                      Bc_flag = bc_flag, $
                      potlin = potlin, iterations = iterations, $
                      Dt = dt, nsave = nsave, double = double, $
                      convergence_level = convergence_level, $
                      pr_iter = pr_iter, $
                      alpha_in = alpha_in, $
                      slow_way = slow_way, $
                      spherical = spherical, $
                      use_fortran = use_fortran, $
                      fortran_exe = fortran_exe, $
                      use_sdf = use_sdf, $
                      clean_divb = clean_divb, $ ;only for uniform cartesia grids so far, and only for the starting field
                      sim_anneal = sim_anneal, $ ;experimental
                      xxx = xxx, _extra = _extra

  Common Bfields, bx, by, bz, pbx, pby, pbz, x, y, z, $
    rsize, tsize, vsize, rsize1, tsize1, vsize1, wf
  Common Vectors, curl_bx, curl_by, curl_bz, div_b, $
    omega_x, omega_y, omega_z, omega2, b2
  message, /info, 'Last Version 17-sep-2007, jmm, jimm@ssl.berkeley.edu'
  print, 'Most recent change: Added use_sdf for I/O'
  print, 'see: http://sprg.ssl.berkeley.edu/~jimm/fff/new_nlfff_version/optimization_fff.html'
;Keywords first
  If(keyword_set(restart_flag)) Then restart_flag = 1 $
  Else restart_flag = 0
  If(keyword_set(bc_flag)) Then bc_flag = 1 $
  Else bc_flag = 0
  If(keyword_set(potlin)) Then potlin = potlin Else potlin = 0
  If(Keyword_set(iterations)) Then iterations = iterations $
  Else iterations = 10000l
  If(keyword_set(dt)) Then dt = dt Else dt = 0.00001
  If(keyword_set(double)) Then dt = double(dt) Else dt = float(dt)
  dt0 = dt
  If(keyword_set(nsave)) Then nsave = nsave Else nsave = 500
  If(keyword_set(convergence_level)) Then afd = convergence_level $
  Else afd = 1.0e-6
  If(keyword_set(pr_iter)) Then pr_iter = pr_iter Else pr_iter = 100
  If(keyword_set(slow_way)) Then slow = 1b Else slow = 0b
  If(keyword_set(spherical)) Then Begin
    print, 'Using Spherical coordinates'
    qsphere = 1
  Endif Else Begin
    print, 'Using Cartesian coordinates'
    qsphere = 0
  Endelse
  If(keyword_set(use_sdf)) Then Begin
    print, 'Using sdf  files for I/O'
    sdfu = 1
    otp_suffix = '.sdf'
  Endif Else Begin
    print, 'Not using sdf  files for I/O'
    sdfu = 0
    otp_suffix = '.dat'
  Endelse
;/* Restart? */
  If (restart_flag Eq 0) Then print, 'Starting a new calculation...' $
  Else print, 'Restarting...'
;/* All BCs? */
  If(bc_flag Eq 0) Then print, 'Only lower BCs available' $
  Else print, 'All boundary conditions are provided'
;/* read in data
  init_data_fff, bx_in, by_in, bz_in, double = double, $
    spherical = spherical, use_sdf = use_sdf, _extra = _extra
  btyp = size(bx, /type)
;/* define numerical grid
  print, 'Defining Numerical Grid'
;get dx,dy,dz
  dx = x[1:*]-x
  dy = y[1:*]-y
  dz = z[1:*]-z
  dx = [dx, dx[rsize-2]]
  dy = [dy, dy[tsize-2]]
  dz = [dz, dz[vsize-2]]
;Get weighting factor
  weightf_fff, _extra = _extra
;/* normalize field
  print, 'Normalizing Field'
  If(keyword_set(spherical)) Then bzmax = max(bx[0, *, *]) $
  Else bzmax = max(bz[*, *, 0])
  If(bzmax Gt 0) Then Begin
    bx = bx/bzmax
    by = by/bzmax
    bz = bz/bzmax
  Endif
;/* calculate potential or linear force-free field, and write it to 
; the file pfield.dat */
  pbx = bx & pby = by & pbz = bz
  alpha = 0.0                   ;to be sure that there's a value
  If(qsphere Eq 1) Then Begin
    If(restart_flag Eq 0) Then Begin
      message, /info, 'Only potential field initialization available'
      pfss_init4fff, _extra = _extra
    Endif
  Endif Else Begin
    If(restart_flag Eq 0) Then Begin
      If(potlin Gt 0) Then Begin
        If(keyword_set(alpha_in)) Then alpha = alpha_in $
        Else alpha = lst_sq_alpha_fff()
        print, 'Least-squares alpha= ', alpha
        If(slow) Then Begin
          calc_lin_fff, dx, dy, alpha, _extra = _extra
        Endif Else call_tomm_fff, alpha
        If(potlin Eq 2) Then Begin
          tbx = pbx & tby = pby & tbz = pbz
;Insert potential field into the pbx, pby, pbz variables
          If(slow) Then Begin
            calc_lin_fff, dx, dy, 0.0, _extra = _extra
          Endif Else call_tomm_fff, 0.0
;pbx, pby, pbz are the potential fields, tbx, tby, tbz are the linear FFF's
          print, 'Combining Linear FFF and Potential field.'        
          r = make_array(type = btyp, value = 0, rsize, tsize, vsize)
          For k = 0, vsize-1 Do For j = 0, tsize-1 Do $
            For i = 0, rsize-1 Do r[i, j, k] = sqrt(x[i]^2+y[j]^2+z[k]^2)
          rmax = max(x)
          pfactor = (r/rmax)^2 < 1.0
          pbx = temporary(tbx)*(1.0-pfactor)+pbx*pfactor
          pby = temporary(tby)*(1.0-pfactor)+pby*pfactor
          pbz = temporary(tbz)*(1.0-pfactor)+pbz*pfactor
        Endif
      Endif Else Begin
        print, 'Calculating Potential FFF'
        alpha = 0.0
        If(slow) Then Begin
          calc_lin_fff, dx, dy, alpha, _extra = _extra
        Endif Else call_tomm_fff, alpha
      Endelse
    Endif
  Endelse
  print, 'Writing Potential or Linear FF Field'
  write_bfield_fff, 'pfield'+otp_suffix, rsize, tsize, vsize, bzmax, $
    pbx, pby, pbz, x, y, z, use_sdf = use_sdf
;/* define initial field. At this point, if i'm restarting, then bx,
;by, and bz are the initial fields, and i need not do anything
  If(restart_flag Eq 0) Then Begin
;If not, then bx, by, bz have not been touched, the 'initial' field
;resides in pbx, pby, pbz, but boundary conditions have not been applied
    print, 'Defining Initial Field'
    initialb_fff, bc_flag, spherical = spherical, _extra = _extra
  Endif
;If clean_divb is set, the try this:
  If(keyword_set(clean_divb)) Then Begin
    If(Not keyword_set(spherical)) Then Begin
      dumb_divclean_proc, bx, by, bz, bx_new, by_new, bz_new, _extra = _extra
      bx = temporary(bx_new) &  by = temporary(by_new) & bz = temporary(bz_new)
    Endif
  Endif
;/* iterate bx, by, bz towards force-free */
  If(keyword_set(sim_anneal)) Then Begin
    sim_anneal_fff, dx, dy, dz, spherical = spherical, _extra = _extra
;/* write field components to file field.dat
    write_bfield_fff, 'field'+otp_suffix, rsize, tsize, vsize, bzmax, $
      bx, by, bz, x, y, z, use_sdf = use_sdf
;/* calculate measures of discrepancy
    print, 'Final discrepancies'
    discrepancy_fff, thetaj, theta, f, force
    print, 'Current-averaged angle=', thetaj
  Endif Else If(keyword_set(use_fortran)) Then Begin
;sniff the version used--this is temporary, and may not last the day
    If(keyword_set(fortran_exe)) Then ffile = fortran_exe $
    Else ffile = 'optimization_fff.exe'
    f90 = strmatch(ffile, '*trial*') Or strmatch(ffile, '*f90*')
    slow = 0                    ;don't do calc_lin_fff again...
    openw, unit, 'fff_input_pars.dat', /get_lun
    printf, unit, dt
    printf, unit, afd
    printf, unit, iterations
    If(f90) Then Begin
      If(qsphere Eq 1) Then printf, unit, '.TRUE.' $
      Else printf, unit, '.FALSE.'
      If(slow Eq 1) Then printf, unit, '.TRUE.' $
      Else printf, unit, '.FALSE.'
    Endif Else Begin
      printf, unit, qsphere
      printf, unit, slow
    Endelse
    printf, unit, alpha
    If(f90 Eq 0) Then printf, unit, sdfu
    free_lun,  unit
    write_bfield_fff, 'field0'+otp_suffix, rsize, tsize, vsize, bzmax, $
          bx, by, bz, x, y, z, use_sdf = use_sdf
    write_scalar_fff, 'weight0'+otp_suffix, rsize, tsize, vsize, wf, $
      use_sdf = use_sdf
    print, 'Spawning: ', ffile
    spawn,  ffile
  Endif Else Begin
    convergence_flag = 0
    If(restart_flag Eq 0) Then openw, lhist_unit, 'lhist.dat', /get_lun $
    Else openw, lhist_unit, 'lhist.dat', /get_lun, /append
    print, 'Starting iterations'	
;/* main loop 
    timei = systime(/sec)
    localdt = 0.0
    i_final = 0
;Initialize saved dV
    For i = 0l, iterations-1l Do Begin
      If(convergence_flag Eq 1) Then Begin
        print, 'It seems to have converged or'
        print, 'is not decreasing for small timesteps'
        Goto, Out_of_loop
      Endif
      If(i Eq 0) Then Begin
;/* calculate objective function for initial field
        print, 'Calculating Objective Function for Initial Field'
; Initialize the curls, divs, omegas, etc.... For each iteration this
; will happen in the evolve_fff routne
        vector_ops_fff, spherical = spherical, _extra = _extra
        l_value = obj_funct_fff(dx, dy, dz, spherical = spherical, $
                                /init_dv, _extra = _extra)
        delta_l = 1.0e20
      Endif
      ii = strtrim(string(i), 2)
      If(i Mod pr_iter Eq 0) Then Begin
        print, i, dt, l_value, delta_l
;/* write interim diagnostics to file
        fmmt = '(i8, e20.6, f16.8)'
        printf, lhist_unit, i, dt, l_value, format = fmmt
      Endif
;/* save to allow restarts */
      If((i Eq 0) Or (i Mod nsave Eq  0)) Then Begin
        write_bfield_fff, 'field'+ii+''+otp_suffix, rsize, tsize, vsize, bzmax, $
          bx, by, bz, x, y, z, use_sdf = use_sdf
      Endif
;clean divergence? every 100 iterations
      If(keyword_set(clean_divb) And keyword_set(xxx) And (i Mod 100 Eq 1)) Then Begin
        If(Not keyword_set(spherical)) Then Begin
          dumb_divclean_proc, bx, by, bz, bx_new, by_new, bz_new, _extra = _extra
          bx = temporary(bx_new) &  by = temporary(by_new) & bz = temporary(bz_new)
          l_value = 1.0e20
        Endif
      Endif
;evolve(dt); /* evolve the field */
      evolve_fff, dx, dy, dz, dt, l_value, local_l, $
        convergence_flag, abs_frac_diff = afd, spherical = spherical, $
        _extra = _extra
      delta_l = (l_value-local_l)/l_value
      If(convergence_flag Eq 0) And (local_l Lt l_value)  $
        Then l_value = local_l
      i_final = i
    Endfor
   
    out_of_loop: dude, ' All done'
    free_lun, lhist_unit
  
    timef = systime(/sec)
    print, 'Time spent on iterative loop, per iteration=', $
      (timef-timei)/i_final
;/* write field components to file field.dat
    write_bfield_fff, 'field'+otp_suffix, rsize, tsize, vsize, bzmax, $
      bx, by, bz, x, y, z, use_sdf = use_sdf
;/* calculate measures of discrepancy
    print, 'Final discrepancies'
    discrepancy_fff, thetaj, theta, f, force
    print, 'Current-averaged angle=', thetaj
  Endelse

  Return
End
