;+
;NAME:
; init_data_fff
;PURPOSE: 
; Sets up initial B field arrays and grids, given bx_in, by_in, bz_in
;CALLING SEQUENCE:
; init_data_fff, bx_in, by_in, bz_in, $
;                   x_in = x_in, y_in = y_in, $
;                   z_in = z_in, double = double, $
;                   spherical = spherical, filename = filename, $
;                   dont_kill_input = dont_kill_input, $
;                   _extra = _extra
;INPUT:
; bx_in, by_in, bz_in = the B field, on the lower boundary
;OUTPUT:
; All in common block bfields
; bx, by, bz = the b field
; rsize = number of x grid points
; tsize = number of grid points in x, y grids
; vsize = number of grid points in z grid
; x, y, z = the grids
;KEYWORDS:
; x_in, y_in, z_in =input grids, otherwise the grids are uniformly
; spaced with vsize=min([tsize, rsize])
; double = if set, make double precision arrays
; spherical = if set, use spherical coordinates
; filename = if set, read the data from this file, note that the input
; b field must be 3d
; dont_kill_input = if set, then the input arrays are not killed
;HISTORY:
; Hacked from main_fff.pro 18-feb-2004, jmm
; bx_in, by_in, bz_in are now explicitly 3d, and x, y no longer
; necessarily have the same number of elements, 17-feb-2005, jmm
; Added common block, reenabled 2-d input fields,added filename
; keyword, jmm, 7-sep-2005
; fixed default grids to match the intention in the documentation, 
; jmm 28-nov-2005
;-
Pro init_data_fff, bx_in, by_in, bz_in, $
                   x_in = x_in, y_in = y_in, $
                   z_in = z_in, double = double, $
                   spherical = spherical, filename = filename, $
                   dont_kill_input = dont_kill_input, $
                   use_sdf = use_sdf, $
                   _extra = _extra

  Common Bfields, bx, by, bz, pbx, pby, pbz, x, y, z, $
    rsize, tsize, vsize, rsize1, tsize1, vsize1

;Read in data, if asked to
  If(keyword_set(filename)) Then Begin 
    init_b_filename = filename
    print, 'Reading Data from: '+filename
    read_bfield_fff, init_b_filename, rsize, tsize, vsize, $
      bx_in, by_in, bz_in, x, y, z, double = double, use_sdf = use_sdf
  Endif
  btyp = size(bx_in, /type)     ;type as an integer
  If(keyword_set(double)) Then Begin
    If(btyp Ne 5) Then Begin    ;convert to double here
      bx_in = double(bx_in)
      by_in = double(by_in)
      bz_in = double(bz_in)
    Endif
  Endif Else Begin              ;float
    If(btyp Ne 4) Then Begin    ;convert to float here
      bx_in = float(bx_in)
      by_in = float(by_in)
      bz_in = float(bz_in)
    Endif
  Endelse
  btyp = size(bx_in, /type)     ;type as an integer
  If(keyword_set(filename)) Then Begin 
    bx = bx_in & by = by_in & bz = bz_in
  Endif Else Begin
;Check size of input
    sxb = size(bx_in)
    rsize = sxb[1]
    tsize = sxb[2]
    If(sxb[0] Eq 2) Then Begin
      If(keyword_set(z_in)) Then Begin
        vsize = n_elements(z_in)
      Endif Else vsize = min([rsize, tsize])
    Endif Else vsize = sxb[3]
    If(Not keyword_set(spherical)) Then Begin
      max_rst = max([rsize, tsize, vsize])
      default_dx = 2.0/(max_rst-1)
    Endif
    bx = make_array(rsize, tsize, vsize, type = btyp, value = 0)
    by = bx
    bz = bx
    If(sxb[0] Eq 2) Then Begin
      If(keyword_set(dont_kill_input)) Then Begin
        bx[*, *, 0] = bx_in
        by[*, *, 0] = by_in
        bz[*, *, 0] = bz_in
      Endif Else Begin
        bx[*, *, 0] = temporary(bx_in)
        by[*, *, 0] = temporary(by_in)
        bz[*, *, 0] = temporary(bz_in)
      Endelse
    Endif Else Begin
      If(keyword_set(dont_kill_input)) Then Begin
        bx = bx_in
        by = by_in
        bz = bz_in
      Endif Else Begin
        bx = temporary(bx_in)
        by = temporary(by_in)
        bz = temporary(bz_in)
      Endelse
    Endelse
;be sure to carry correct data type
    x = make_array(rsize, type = btyp, value = 0)
    y = make_array(tsize, type = btyp, value = 0)
    z = make_array(vsize, type = btyp, value = 0)
    If(n_elements(x_in) Eq rsize) Then x[*] = x_in Else Begin
      If(keyword_set(spherical)) Then Begin ;1 to 2
        dx = 1.0d0/(rsize-1)
        x[*] = dx*findgen(rsize)+1.0
      Endif Else Begin          ;-1 to 1 for maximum grids!
        offset = -default_dx*float(rsize/2)
        x[*] = default_dx*findgen(rsize)+offset
      Endelse
    Endelse
    If(n_elements(y_in) Eq tsize) Then y[*] = y_in Else Begin
      If(keyword_set(spherical)) Then Begin ;0 to pi
        dy = !pi/(tsize-1)
        y[*] = dy*findgen(tsize)
      Endif Else Begin          ;-1 to 1
        offset = -default_dx*float(tsize/2)
        y[*] = default_dx*findgen(tsize)+offset
      Endelse
    Endelse
    If(n_elements(z_in) Eq vsize) Then z[*] = z_in Else Begin
      If(keyword_set(spherical)) Then Begin ;0 to 2pi
        dz = 2.0*!pi/(vsize-1)
        z[*] = dz*findgen(vsize)
      Endif Else Begin          ;0 to 2;the pixels need to be the same size
        z[*] = default_dx*findgen(vsize)
      Endelse
    Endelse
  Endelse
;Other stuff
  rsize1 = rsize-1 & tsize1 = tsize-1 & vsize1 = vsize-1
  Return
End
