;+
Pro fff_temp_pre, bx_in, by_in, bz_in, x_in, y_in, bx, by, bz, $
                  mask = mask, noplot = noplot, $
                  quiet = quiet, verbose = verbose, $
                  xegoal = xegoal, $
                  yegoal = yegoal, $
                  zegoal = zegoal, $
                  m3x = m3xfactor, $
                  m3y = m3yfactor, $
                  m3z = m3zfactor, $
                  m4 = m4factor, $
                  fixm3 = fixm3, $
                  mollifier = mollifier, $
                  converge = converge, $
                  xerror = xerror, $
                  yerror = yerror, $
                  zerror = zerror, $
                  nplot = nplot, $
                  _extra = _extra

;NAME:
;     fff_temp_pre
;PURPOSE:
;     Preprocess a magnetogram to make it closer to force-free and
;     torque-free while keeping the field close to the original.
;CALLING SEQUENCE:
;     jp = bpreprocess(bx_in, by_in, bz_in, x_in, y_in, bx, by, bz)
;INPUTS:
;     bx_in, by_in, bz_in = ambiguity resolved magnetic field
;KEYWORDS
;     [xyz]egoal = Magnetic field error goal (Gauss).  The algorithm will 
;                  try to make the errors resulting from reducing the force
;                  and torque equal to this value.  Default = [50, 50, 20]
;     [xyz]error = on output, set to the final average field error
;                  attained. 
;     m3[xyz] = multiplicitive factor to adjust the weight on the B error
;               term.  On outut set to the final value used.
;     m4 = multiplicitive factor to adjust the weight on the smoothing
;               term.  Default = 1.0.  Higher values give more
;               smoothing.  Zero turns the smoothing off.
;     /fixm3 = keep m3 fixed, do not iteratively adjust it.  If this
;              is set, egoal is ignored.  Using m3 and /fixm3 to set
;              the final error could be more robust since m3 is not
;              iterated.  But it would require more effort to get the
;              field errors you want, but converges faster.
;     mollifier = controls how fast the iteration proceeds.  Default
;                 is 0.1.  If this is too fast, try 0.01 to keep the
;                 iteration from diverging.  If the iteration is
;                 proceeding to slowly, increase this number
;                 (1.0,10.0?) 
;     converge = frational convergence criteria.  Defaul = 1.e-5
;     mask = index of points to include in the calculation.  Default
;            is to use all points.
;     /noplot = do not plot intermediate results.
;     nplot = plot after every nplot iterations.  Default is 25 so
;             that the plots are done every 25th iteration.
;     /quiet = work quietly.
;     /verbose = work loudly.
;OUTPUTS:
;     bx, by, bz = preprocessed magnetic field.
;PROCEDURE:
;     As described in Wiegelmann et al., Solar Phys. 233, 215, 2006.
;     There is an error in the dL1/dB terms in the paper which is
;     corrected here. Also, the errors on Bx, By, and Bz are treated
;     independently, which is more general than the Wiegelmann
;     description. 
;MODIFICATION HISTORY:
;     T. Metcalf 2006-Apr-30 Initial version.
;     TRM 2006-May-31 Fixed bug in the bit of code that checks for
;                     increasing L.  Also update b[xyz]laplace when
;                     j.b[xyz] is updated.
;     TRM 2006-Jun-01 Changed the field errors to sqrt(L3/area), from
;                     L3/area. Added nplot keyword.  Use sig_b[xyz]h
;                     to set error goals when they are not specified
;                     by the user.  Use Lcalib insted of L in the
;                     convergence check.  Limit the change to the
;                     mollifier to a factor of 1000.
;     jmm, 6-jul-2006, Hacked from Tom Metcalf's bpreprocess.pro, for
;     use when you don't have an IVM magnetogram structure, or any
;     solar coordinate information
;-
;Initialize b
  bx = bx_in & by = by_in & bz = bz_in

  if n_elements(m3xfactor) NE 1 then m3xfactor = 1.0
  if n_elements(m3yfactor) NE 1 then m3yfactor = 1.0
  if n_elements(m3zfactor) NE 1 then m3zfactor = 1.0
  if n_elements(m4factor) NE 1 then m4factor = 1.0
  if n_elements(mollifier) NE 1 then mollifier = 0.1
  if NOT keyword_set(converge) then converge = 1.e-5
  if NOT keyword_set(nplot) then nplot = 25L else nplot = long(nplot) > 1L

;set up coordinate system, x, y, and dA are 2d, jmm, 6-jul-2006
  nx = n_elements(x_in)
  ny = n_elements(y_in)
  dx = x_in[1:*]-x_in
  dy = y_in[1:*]-y_in
  dx = [dx, dx[nx-2]]
  dy = [dy, dy[ny-2]]
  dxp = fltarr(nx, ny) &  dyp = dxp
  For i = 0, nx-1 Do dxp[i, *] = dx[i]
  For j = 0, ny-1 Do dyp[*, j] = dy[j]
  dA = dxp*dyp
  For i = 0, nx-1 Do dxp[i, *] = x_in[i]
  For j = 0, ny-1 Do dyp[*, j] = y_in[j]
  x = temporary(dxp) & y = temporary(dyp)

  if NOT keyword_set(xegoal) then xegoal = 50.0
  if NOT keyword_set(yegoal) then yegoal = 50.0
  if NOT keyword_set(zegoal) then zegoal = 20.0

  if NOT keyword_set(mask) then begin
    mask = lindgen(nx, ny)
  endif
  umask = bytarr(nx, ny)
  umask[mask] = 1b
  umask = where(umask NE 1b, numask)

  D2 = max(x_in-x_in[0])*max(y_in-y_in[0])  ;size scale of the box
  B2avg = (total(sqrt(bx^2+by^2+bz^2))/(nx*ny))^2 ;average total field squared

  area = total(dA)

  if numask GT 0 then dA[umask] = 0.0 ; easy way to turn off pixels

  Larray = [-1.0d0]
  Exarray = [-1.0d0]
  Eyarray = [-1.0d0]
  Ezarray = [-1.0d0]
  Farray = [-1.0d0]

  m = mollifier
  m1 = 1.0d0/(B2avg*area^2)     ; intial multipliers.  Sets units.
  m2 = m1/D2
  m3x = double(m3xfactor)/area
  m3y = double(m3yfactor)/area
  m3z = double(m3zfactor)/area
  m4 = double(m4factor)/area

  L = -1.0
  Lcalib = L
  F = -1.0
  done = 0
  nincrease = 0L
  oldm1 = m1
  oldm2 = m2
  oldm3x = m3x
  oldm3y = m3y
  oldm3z = m3z
  oldm4 = m4

;Mehrstellen stencil, no slower than the 5-point stencil,
;but errors are isotropic, for whatever that is worth.
  kernel = double([[1, 4, 1], [4, -20, 4], [1, 4, 1]])/6.0d0  
;kernel = double([[0,1,0],[1,-4,1],[0,1,0]]) ; Standard 5-point
;stencil for the Laplacian operator
  bxlaplace = CONVOL(bx, kernel, /CENTER, /EDGE_TRUNCATE) ; Laplacian of Bx
  bylaplace = CONVOL(by, kernel, /CENTER, /EDGE_TRUNCATE)
  bzlaplace = CONVOL(bz, kernel, /CENTER, /EDGE_TRUNCATE)

  REPEAT begin
;iterate to the minimum L.  See Wiegelmann et al.
    dLdBx = 2.0*m1*(total(bx*bz*dA, /double, /nan)*bz*dA - $
                    2.0*total(dA*(bz^2-bx^2-by^2), /double, /nan)*bx*dA ) - $
      2.0*m2*(2.0*total(dA*x*(bz^2-bx^2-by^2), /double, /nan)*x*bx*dA + $
              2.0*total(dA*y*(bz^2-bx^2-by^2), /double, /nan)*y*bx*dA - $
              total(dA*(y*bx*bz-x*by*bz), /double, /nan)*y*bz*dA  ) + $
      2.0*m3x*(bx-bx_in)*dA + $
      2.0*m4*CONVOL(bxlaplace, kernel, /CENTER, /EDGE_TRUNCATE)*dA

    dLdBy = 2.0*m1*(total(by*bz*dA, /double, /nan)*bz*dA - $
                    2.0*total(dA*(bz^2-bx^2-by^2), /double, /nan)*by*dA ) - $
      2.0*m2*(2.0*total(dA*x*(bz^2-bx^2-by^2), /double, /nan)*x*by*dA + $
              2.0*total(dA*y*(bz^2-bx^2-by^2), /double, /nan)*y*by*dA + $
              total(dA*(y*bx*bz-x*by*bz), /double, /nan)*x*bz*dA  ) + $
      2.0*m3y*(by-by_in)*dA + $
      2.0*m4*CONVOL(bylaplace, kernel, /CENTER, /EDGE_TRUNCATE)*dA

    dLdBz = 2.0*m1*(total(bx*bz*dA, /double, /nan)*bx*dA + $
                    total(by*bz*dA, /double, /nan)*by*dA + $
                    2.0*total(dA*(bz^2-bx^2-by^2), /double, /nan)*bz*dA) + $
      2.0*m2*(2.0*total(dA*x*(bz^2-bx^2-by^2), /double, /nan)*x*bz*dA + $
              2.0*total(dA*y*(bz^2-bx^2-by^2), /double, /nan)*y*bz*dA + $
              total(dA*(y*bx*bz-x*by*bz), /double, /nan)*(y*bx-x*by)*dA ) + $
      2.0*m3z*(bz-bz_in)*dA + $
      2.0*m4*CONVOL(bzlaplace, kernel, /CENTER, /EDGE_TRUNCATE)*dA

    bxold = bx
    byold = by
    bzold = bz
    bxlaplaceold = bxlaplace
    bylaplaceold = bylaplace
    bzlaplaceold = bzlaplace
    Fold = F
    Lold = L

    bx = bx - m*dLdBx
    by = by - m*dLdBy
    bz = bz - m*dLdBz

    bxlaplace = CONVOL(bx, kernel, /CENTER, /EDGE_TRUNCATE) ; Laplacian of Bx
    bylaplace = CONVOL(by, kernel, /CENTER, /EDGE_TRUNCATE)
    bzlaplace = CONVOL(bz, kernel, /CENTER, /EDGE_TRUNCATE)

    L1 = (total(bx*bz*dA, /double, /nan))^2 + $
      (total(by*bz*dA, /double, /nan))^2 + $
      (total((bz^2-bx^2-by^2)*dA, /double, /nan))^2 ; Force-free condition

    L2 = ((total(x*(bz^2-bx^2-by^2)*dA, /double, /nan))^2 + $
          (total(y*(bz^2-bx^2-by^2)*dA, /double, /nan))^2 + $
          (total((y*bx*bz - x*by*bz)*dA, /double, /nan))^2) ; torque free condition

    L3x = total(dA*(bx-bx_in)^2, /double, /nan) ; error on Bx
    L3y = total(dA*(by-by_in)^2, /double, /nan) ; error on By
    L3z = total(dA*(bz-bz_in)^2, /double, /nan) ; error on Bz
    L4 =  total(dA*(bxlaplace^2+bylaplace^2+bzlaplace^2), /double, /nan) ; Smoothness condition
;if L increased, do not take the step, but reduce the mollifier
    if L GE 0.0 AND (oldm1*L1+oldm2*L2+oldm3x*L3x+oldm3y*L3y+oldm3z*L3z+oldm4*L4 GT Lold*1.10) then begin
      bx = bxold
      by = byold
      bz = bzold
      bxlaplace = bxlaplaceold
      bylaplace = bylaplaceold
      bzlaplace = bzlaplaceold
      m = m/1.2
      nincrease = nincrease + 1L
      if nincrease GT 100L then done = 1 ; If too many increases, quit
      if done then message, /info, 'ERROR: L is increasing.  Giving up.' $
      else if keyword_set(verbose) then print, 'reducing mollifier'
    endif else begin
      nincrease = 0L
      L = m1*L1 + m2*L2 + m3x*L3x + m3y*L3y + m3z*L3z + m4*L4
      F = m1*L1 + m2*L2         ; force and torque only
      if keyword_set(verbose) then print, m1*L1, m2*L2, m3x*L3x, m3y*L3y, m3z*L3z, m4*L4
      Larray = [Larray, L]
      Exarray = [Exarray, sqrt(L3x/area)]
      Eyarray = [Eyarray, sqrt(L3y/area)]
      Ezarray = [Ezarray, sqrt(L3z/area)]
      Farray = [Farray, F]
      n = n_elements(Larray)
      if n EQ 2 then begin      ; only do this once on the first pass
        Lstart = L
        L1start = L1
        L2start = L2
        L3xstart = L3x
        L3ystart = L3y
        L3zstart = L3z
        L4start = L4
        m3xstart = m3x
        m3ystart = m3y
        m3zstart = m3z
        Fstart = F
      endif
      Lcaliblast = Lcalib
      Lcalib = m1*L1 + m2*L2 + m3xstart*L3x + m3ystart*L3y + m3zstart*L3z + m4*L4 ; w/o m3 variation

      oldm1 = m1
      oldm2 = m2
      oldm3x = m3x
      oldm3y = m3y
      oldm3z = m3z
      oldm4 = m4

      if NOT keyword_set(fixm3) then begin
        if sqrt(L3x/area) GE xegoal*1.1 then m3x = m3x*1.03 ; adjust weight of error term
        if sqrt(L3x/area) LE xegoal*0.9 then m3x = m3x/1.01
        if sqrt(L3y/area) GE yegoal*1.1 then m3y = m3y*1.03 ; adjust weight of error term
        if sqrt(L3y/area) LE yegoal*0.9 then m3y = m3y/1.01
        if sqrt(L3z/area) GE zegoal*1.1 then m3z = m3z*1.03 ; adjust weight of error term
        if sqrt(L3z/area) LE zegoal*0.9 then m3z = m3z/1.01
; speed up if error is low, slow down if it is high
        if sqrt(L3x/area) LE xegoal*0.9 OR sqrt(L3y/area) LE yegoal*0.9 OR sqrt(L3z/area) LE zegoal*0.9 then $
          m = (m*1.01) < (mollifier*1000.)
        if sqrt(L3x/area) GT xegoal*1.1 OR sqrt(L3y/area) GT yegoal*1.1 OR sqrt(L3z/area) GT zegoal*1.1 then $
          m = (m/1.01) > (mollifier/1000.)
      endif

      if n GT 2 then begin
        done = (abs(Lcalib-Lcaliblast)/Lcalib LT converge AND $
;abs(Larray[n-1]-Larray[n-2])/Larray[n-1] LT converge AND $
          abs(Exarray[n-1]-Exarray[n-2])/Exarray[n-1] LT converge AND $
          abs(Eyarray[n-1]-Eyarray[n-2])/Eyarray[n-1] LT converge AND $
          abs(Ezarray[n-1]-Ezarray[n-2])/Ezarray[n-1] LT converge AND $
          n GT 100L) OR n GT 100000L
        if NOT keyword_set(noplot) AND ((n MOD nplot) eq 0 OR done) then begin
          pmulti = !p.multi
          !p.multi = [0, 1, 5, 0]
          plot, Farray[1:*], title = 'Force+Torque '+string(Farray[n-1]), /xstyle, /ystyle, charsize = 2.0, /ylog
          oplot, [0, n], [Farray[1], Farray[1]], line = 1
          plot, Larray[1:*], title = 'L '+string(Larray[n-1]), /xstyle, /ystyle, charsize = 2.0, /ylog
          oplot, [0, n], [Lstart, Lstart], line = 1
          oplot, [0, n], [Lcalib, Lcalib], line = 2
          plot, Exarray[1:*], title = 'Bx error '+string(Exarray[n-1])+' '+string(m3x*area), /xstyle, /ystyle, charsize = 2.0
          oplot, [0, n], [xegoal, xegoal], line = 1
          plot, Eyarray[1:*], title = 'By error '+string(Eyarray[n-1])+' '+string(m3y*area), /xstyle, /ystyle, charsize = 2.0
          oplot, [0, n], [yegoal, yegoal], line = 1
          plot, Ezarray[1:*], title = 'Bz error '+string(Ezarray[n-1])+' '+string(m3z*area), /xstyle, /ystyle, charsize = 2.0
          oplot, [0, n], [zegoal, zegoal], line = 1
          empty
          !p.multi = pmulti
        endif
      endif else done = 0
    endelse

  endrep UNTIL done

  if keyword_set(verbose) then begin
    print, strcompress(string(n)+' iterations.')
    print, 'L start:  ', Lstart
    print, 'L finish: ', L
    print, 'm3x start:  ', m3xstart
    print, 'm3x finish: ', m3x
    print, 'm3y start:  ', m3ystart
    print, 'm3y finish: ', m3y
    print, 'm3z start:  ', m3zstart
    print, 'm3z finish: ', m3z
  endif

  if F GT Fstart then begin
    message, /info, 'WARNING: F increased.  Returning original data.'
    return
  endif

  m3xfactor = m3x*area
  m3yfactor = m3y*area
  m3zfactor = m3z*area
  xerror = sqrt(L3x/area)
  yerror = sqrt(L3y/area)
  zerror = sqrt(L3z/area)
    
  return

end
