function poisson, infield, dx, inspect=inspect, outspect=outspect

; this program solves poission's equation:
; the laplacian of the 2-d scalar output 
; field gives the 2-d scalar input field
;
; Approach: for each possible value of (k_x, k_y), find output
; FFT spectrum by dividing input FFT spectrum by (k_x^2 + k_y^2)

if (n_params() eq 0) then begin
    message,' NAME: POISSON.PRO (an IDL function)',/info
    message,'',/info
    message,' PURPOSE: Returns 2D soln. of horizontal Poisson equation,',/info
    message,'    given 2D input array: laplacian(solution) = input.',/info
    message,'    Periodic approach: takes FFT of input, divides resulting', $
      /info
    message,'    spectrum by (k_x^2 + k_y^2), then takes inverse FFT.',/info
    message,'    If input is double precision, output is, too.',/info
    message,'',/info
    message,' USAGE: poisson_result = poisson(input_arr, pixel_size)',/info
    message,'',/info
    message,' REQUIRED: INFIELD = 2D input array, either float or double.', $
      /info
    message,'',/info
    message,' OPTIONAL KEYWORDS: DX = pixel size; default = 1e+0 (float)',/info
    message,'    INSPECT = set to variable to contain FFT of input array',/info
    message,'    OUTSPECT = set to variable to contain FFT of output array', $
      /info
    message,'',/info
    message,' ERROR HANDLING: Primitive; halts mid-function.',/info
    message,'',/info
    message,' HISTORY: Written Brian T. Welsch, SSL-UCB - c. fall, 2004?',/info
    message,'    BTW: Incorporate double precision.',/info
    message,'    BTW: Last Modified 09-Jan-2008.',/info
    return,0
endif 

if not(keyword_set(dx)) then begin
    dx = 1.
    message,'No dx passed, assuming unit pixel size.',/info
endif

dims = size(infield)

nx = dims(1)	; define dimensionality
ny = dims(2)

; Define arrays, either double or single precision, based upon input
;===================================================================
if (dims(3) eq 5) then begin
    outfield = dblarr(nx,ny)     ; this is output array
    outspect = dcomplexarr(nx,ny) ; this spectrum w/FFT gives outfield
endif else begin
    outfield = fltarr(nx,ny)     ; this is output array
    outspect = complexarr(nx,ny) ; this spectrum w/FFT gives outfield
endelse

; Define frequency arrays as IDL does -- for even and odd cases
;================================================================
if ((nx)mod(2) eq 0) then xfreqs = $
[0,(findgen(nx/2) + 1)/(nx),-reverse(findgen(nx/2-1) + 1)/(nx)] $
else xfreqs = $
[0,(findgen(nx/2.-.5) + 1)/(nx),-reverse(findgen(nx/2.-.5) + 1)/(nx)]


if ((ny)mod(2) eq 0) then yfreqs = $
[0,(findgen(ny/2) + 1)/(ny),-reverse(findgen(ny/2-1) + 1)/(ny)] $
else yfreqs = $
[0,(findgen(ny/2.-.5) + 1)/(ny),-reverse(findgen(ny/2.-.5) + 1)/(ny)]

inspect = fft(infield, -1)

; Transform, divide by (kx^2 + ky^2), transform back.
;=======================================================
for i = 0,nx-1 do begin
	for j = 0,ny-1 do begin

		; laplacian of 0-modes is zero, so start w/non-zero modes.
		if ((i ne 0) or (j ne 0)) then $
		outspect(i,j) = -(2*!pi)^(-2)/ $
				((xfreqs(i))^2+(yfreqs(j))^2) $
				*inspect(i,j)

	endfor
endfor

outfield = real_part(fft(outspect,1))

; Scale derivatives by pixel size.
;===================================
outfield = outfield*dx^2

return, outfield

end



