;+
;NAME:
; dumb_divclean_proc
;PURPOSE:
; clean the divergence from a 3d cartesian vector, as if it were a
; magnetic field
;CALLING SEQUENCE:
; dumb_divclean_proc, bx, by, bz, bx_new, by_new, bz_new
;INPUT:
; bx, by, bz = are the x,y,z components of the B field
;OUTPUT:
; bx_new, by_new, bz_new = the "cleaned" B field
;KEYWORDS:
; verbose = if set, lots of diagnostic output
;HISTORY:
; Hacked from Brain Welsch's dumb_divclean.pro,
; 4-mar-2008, jmm, jimm@ssl.berkeley.edu
;-
Pro dumb_divclean_proc, bx, by, bz, bx_new, by_new, bz_new, $
                        verbose = verbose, nz_clean = nz_clean, _extra = _extra


; Set some parameters
;================================================
  dx = sqrt(2.0)       ; uniform grid, but use non-trivial pixel size 

  nx = n_elements(bx[*, 0, 0]) 
  ny = n_elements(bx[0, *, 0]) 
  nz = n_elements(bx[0, 0, *]) 
  If(keyword_set(nz_clean)) Then nzc = nz_clean Else nzc = 10

; Easiest way to define "cleaned" arrays, 
; and transfer type & wall values to them
;===========================================
  bx_new = bx           
  by_new = by
  bz_new = bz

  t_init = !stime
  t0 = systime(/sec)

; Compute divergence of input vector field on uniform, 3D grid
  div_check, bx, by, bz, divb, verbose=verbose, dx = dx

; First, fix B_z in interior layers adjacent to bottom & top
;================================================

; First, correct B_z in first zone above bottom
  bz_new[*, *, 1] = 0.75*bz[*, *, 0] + 0.25*bz[*, *, 2] $
    - dx*xderiv_fd(bx[*, *, 0], dx)/2. - dx*yderiv_fd(by[*, *, 0], dx)/2.

; Next, correct B_z in last zone before top
  bz_new[*, *, nz-2] = 0.75*bz[*, *, nz-1] + 0.25*bz[*, *, nz-3] $
  + dx*xderiv_fd(bx[*,*,nz-1],dx)/2. + dx*yderiv_fd(by[*,*,nz-1],dx)/2.

; Correct horizontal divergence of horizontal B on interior layers 
for i=1,nzc-1 do begin

    divb_i = divb[*,*,i]

    ; Can't solve Poisson's equation on outer three zones, so crop 'em.
    divb_crop_i = divb_i[3:nx-4, 3:ny-4] ; crop one zone from ea. side
    divb_crop_input = divb_crop_i
    chi_crop_input = poisson(divb_crop_i, dx) ; use FFT for init. guess

    ; Poisson solver won't update outer two zones, so make them zeroes.
    for j = 0, 1 do begin       ;
      chi_crop_input = pad_array(chi_crop_input, /silent)
      divb_crop_input = pad_array(divb_crop_input, /silent)
    endfor

    chi_crop_i = chi_crop_input
    eps = 1e-6
    poissond_qd, divb_crop_input, chi_crop_i, ds = dx, verbose = 0, eps = eps

    chi_i = fltarr(nx, ny)
    chi_i[1:nx-2,1:ny-2] = chi_crop_i[0:nx-3,0:ny-3]

    dchi_dx = xderiv_fd(chi_i, dx)
    dchi_dy = yderiv_fd(chi_i, dx)

    dbx = fltarr(nx, ny)
    dby = fltarr(nx, ny)
    
    dbx[3:nx-4, 3:ny-4] = -dchi_dx[3:nx-4, 3:ny-4]
    dby[3:nx-4, 3:ny-4] = -dchi_dy[3:nx-4, 3:ny-4]

    bx_new[*, *, i] = bx_new[*, *, i] + dbx 
    by_new[*, *, i] = by_new[*, *, i] + dby
    
;    soln_i = (xderiv_fd(xderiv_fd(chi_i,dx),dx)+ $
;              yderiv_fd(yderiv_fd(chi_i,dx),dx))
;    window,1,xs=400,ys=400
;    plot,divb_i,soln_i,psym=3
;    r_corr_i = r_correlate(divb_i,soln_i)
;    print,'Correlations:',correlate(divb_i,soln_i),r_corr_i(0)
;    stop

    if (keyword_set(verbose)) then print, 'Solved for chi on layer #', i
    
  endfor

; Check to make sure interior calcs actually improved things...
;div_check, bx_new, by_new, bz_new, divb_new, verbose=verbose, dx=dx

; Next, fix the walls! First xmin & xmax, then ymin & ymax
;===========================================================
  dbz_dz_3d = zderiv_fd3d(bz_new, dx) ; use Bz_new!
  dby_dy_3d = yderiv_fd3d(by_new, dx) ; By_new not necess for walls, but anyway...

; First, correct B_x in first zone inward of left wall (old by is okay)
  bx_new[1, *, 1:nz-2] = 0.75*bx[0, *, 1:nz-2] + 0.25*bx[2, *, 1:nz-2] $
    - dx*dby_dy_3d[0, *, 1:nz-2]/2. - dx*dbz_dz_3d[0, *, 1:nz-2]/2. 

; Next, correct B_x in last zone before right wall
  bx_new[nx-2, *, 1:nz-2] = 0.75*bx[nx-1, *, 1:nz-2] + 0.25*bx[nx-3, *, 1:nz-2] $
    + dx*dby_dy_3d[nx-1, *, 1:nz-2]/2. + dx*dbz_dz_3d[nx-1, *, 1:nz-2]/2.

  dbx_dx_3d = xderiv_fd3d(bx_new, dx) ; Bx_new not necess for walls, but anyway...

; Onto B_y!  Leave changed x-values alone, though
; Correct the first zone inward of y=0 wall...
  by_new[2:nx-3, 1, 1:nz-2] = 0.75*by[2:nx-3, 0, 1:nz-2] + 0.25*by[2:nx-3, 2, 1:nz-2] $
    - dx*dbx_dx_3d[2:nx-3, 0, 1:nz-2]/2. - dx*dbz_dz_3d[2:nx-3, 0, 1:nz-2]/2. 

; ...finally, correct B_y in last zone before max(y) wall
  by_new[2:nx-3, ny-2, 1:nz-2] = 0.75*by[2:nx-3, ny-1, 1:nz-2] + 0.25*by[2:nx-3, ny-3, 1:nz-2] $
    + dx*dbx_dx_3d[2:nx-3, ny-1, 1:nz-2]/2. + dx*dbz_dz_3d[2:nx-3, ny-1, 1:nz-2]/2.

; Check to make sure wall calcs actually improved things...
  div_check, bx_new, by_new, bz_new, divb_new, verbose = verbose, dx = dx

  t_final = !stime
  t1 = systime(/sec)

  print, '===================================================================='
  print, ' Initial & final times:'
  print, t_init
  print, t_final
  print, '===================================================================='
  print, 'Total time: ', (t1-t0)/60.0, ' minutes'

end

