;+
;NAME:
; div_rtp
;PURPOSE:
; Does the divergence for a 3d vector, using the DERIV routine,
; in Cartesian coordinates
;CALLING SEQUENCE:
; div_rtp, ax, ay, az, x, y, z, div_a
;INPUT:
; ax, ay, az = are 3d arrays defined on (x,y,z)
; x, y, z, = the coordinates on which the arrays is defined.
;OUTPUT:
; div_a = the divergence of a
;HISTORY:
; 18-aug-2000, jmm, jimm@ssl.berkeley.edu
; switched to use deriv_dx, _dy, and _dz, 
; Added _extra, 1-apr-2003, jmm
; spherical q's x -> r, y -> theta, z -> phi, 21-jan-2004, jmm
;-
PRO div_rtp, ax, ay, az, x, y, z, div_a, $
             use_saved_grids = use_saved_grids, _extra = _extra

   Common saved_rtp, rx, sintheta
   sizax = size(ax)
   IF(sizax[0] NE 3) THEN BEGIN
      message, /info, 'Array must be 3-d'
      RETURN
   ENDIF
   sizay = size(ay)
   IF(sizay[0] NE 3) THEN BEGIN
      message, /info, 'Array must be 3-d'
      RETURN
   ENDIF
   sizaz = size(az)
   IF(sizaz[0] NE 3) THEN BEGIN
      message, /info, 'Array must be 3-d'
      RETURN
   ENDIF
   IF(total(sizax[0:3]) NE total(sizay[0:3]) OR $
      total(sizax[0:3]) NE total(sizaz[0:3])) THEN BEGIN
      message, /info, 'Arrays have mismatched sizes'
      RETURN
   ENDIF
   
   nx = N_ELEMENTS(x)
   IF(Nx NE sizax[1]) THEN BEGIN
      message, /info, 'Bad X array'
      RETURN
   ENDIF
   ny = N_ELEMENTS(y)
   IF(Ny NE sizax[2]) THEN BEGIN
      message, /info, 'Bad Y array'
      RETURN
   ENDIF
   nz = N_ELEMENTS(z)
   IF(Nz NE sizax[3]) THEN BEGIN
      message, /info, 'Bad Z array'
      RETURN
   ENDIF

   If(n_elements(rx) Eq 0 Or (Not keyword_set(use_saved_grids))) Then Begin
       rx = rebin(x, nx, ny, nz)
       sintheta = rebin(sin(y), ny, nz, nx)
       sintheta = transpose(sintheta, [2, 0, 1])
   Endif
   rx2 = rx^2
   daxdx = deriv_dx(x, rx2*ax, _extra = _extra)
   daydy = deriv_dy(y, sintheta*ay, _extra = _extra)
   dazdz = deriv_dz(z, az, _extra = _extra)

   div_a = (daxdx/rx2)+(daydy/(rx*sintheta))+(dazdz/(rx*sintheta))

   RETURN
END

