;+
; NAME:
;	Deriv_dz
; PURPOSE:
;       Partial Derivative of a 3-d array with respect to the second
;       (z) index
; CALLING SEQUENCE:
;	Dfdz = Deriv_dz(z, f, use_saved_grids=use_saved_grids)
; INPUTS:
;	Z:  Variable to differentiate with respect to
;	f:  Function to be differentiated, must be 3-d with the first
;	index having the same number of elements as Z
; OUTPUTS:
;	dfdz= the derivative df/dz at each point
; KEYWORDS:
;       use_saved_grids, if set then assume that the z grid hasn't
;       changed since the last call, and use the same matrices
; COMMON BLOCKS:
;	deriv_dz_saved, holds the coefficient matrices for the grid
; HISTORY:
;	1-apr-2003, jmm, jimm@ssl.berkeley.edu, from deriv.pro
;       24-may-2005, jmm, z01, z02, z12 need not be saved, sz's and
;       rz's can be scalars
;-
Function Deriv_dz, z, f, use_saved_grids = use_saved_grids, _extra = _extra


;Saved grids
  common deriv_dz_saved, cz1, cz2, cz3, sz1, sz2, sz3, rz1, rz2, rz3

;Error handler
  err_dfdz = 0
  catch, err_dfdz
  If(err_dfdz Ne 0) Then Begin
    catch, /cancel
    Print, 'Error'
    help, /last_message, output = theerrormessage
    For j = 0, n_elements(theerrormessage)-1 Do print, theerrormessage[j]
    print, 'Returning -1'
    Return, -1
  Endif
  n = n_elements(z)
  n2 = n-2

  sizf = size(f)
  If(sizf[0] Ne 3) Then message, 'F Array must be 3-d'
  nx = sizf[1] & ny = sizf[2] & nz = sizf[3]

  If(n Lt 3) Then message, 'Parameters must have at least 3 points'
  If(n ne nz) Then message, 'Mismatch between z and f[0,0,*]'

;df/dx=y0*(2x-x1-x2)/(x01*x02)+y1*(2x-x0-x2)/(x10*x12)+y2*(2x-x0-x1)/(x20*x21)
;Where: x01 = x0-x1, x02 = x0-x2, x12 = x1-x2, etc.
  If(Not Keyword_set(use_saved_grids)) Then Begin
    init: type = size(z, /type)    ;If not floating type, ensure floating...
    If((type Ne 4) And (type Ne 5) And (type Ne 6) And (type Ne 9)) Then Begin
      zz = float(z)
      z12 = zz - shift(zz, -1)  ;z1 - z2
      z01 = shift(zz, 1) - zz   ;z0 - z1
      z02 = shift(zz, 1) - shift(zz, -1) ;z0 - z2
    Endif Else Begin            ;Already floating or double
      z12 = z - shift(z, -1)    ;z1 - z2
      z01 = shift(z, 1) - z     ;z0 - z1
      z02 = shift(z, 1) - shift(z, -1) ;z0 - z2
    Endelse
    cz1 = (z12 / (z01*z02))
    cz2 = (1./z12 - 1./z01)
    cz3 = (z01 / (z02 * z12))
    sz1 = (z01[1]+z02[1])/(z01[1]*z02[1])
    sz2 = z02[1]/(z01[1]*z12[1])
    sz3 = z01[1]/(z02[1]*z12[1])
    rz1 = z12[n2]/(z01[n2]*z02[n2])
    rz2 = z02[n2]/(z01[n2]*z12[n2])
    rz3 = (z02[n2]+z12[n2]) / (z02[n2]*z12[n2])
;Replicate to size of f
    cz1 = transpose(rebin(cz1, nz, ny, nx, /sample))
    cz2 = transpose(rebin(cz2, nz, ny, nx, /sample))
    cz3 = transpose(rebin(cz3, nz, ny, nx, /sample))
  Endif Else Begin
    If(n_elements(cz1) Eq 0) Then Goto, init
    If(n_elements(cz1[0, 0, *]) Ne nz) Then Goto, init
  Endelse
;Do the derivative
  d = f & d[*] = 0.0
  d = shift(f, 0, 0, 1) * cz1 + $ ;Middle points
    f * cz2 - $
    shift(f, 0, 0, -1) * cz3
; Formulae for the first and last points:
  d[*, *, 0] = f[*, *, 0] * sz1 - $ ;First point
      f[*, *, 1] * sz2 + $
      f[*, *, 2] * sz3
  d[*, *, n-1] = -f[*, *, n-3] * rz1 + $ ;Last point
    f[*, *, n-2] * rz2 - $
    f[*, *, n-1] * rz3
  
  Return, d
End
