;+
; NAME:
;	Deriv_dy
; PURPOSE:
;       Partial Derivative of a 3-d array with respect to the second
;       (y) index
; CALLING SEQUENCE:
;	Dfdy = Deriv_dy(y, f, use_saved_grids=use_saved_grids)
; INPUTS:
;	Y:  Variable to differentiate with respect to
;	f:  Function to be differentiated, must be 3-d with the first
;	index having the same number of elements as Y
; OUTPUTS:
;	dfdy= the derivative df/dy at each point
; KEYWORDS:
;       use_saved_grids, if set then assume that the y grid hasn't
;       changed since the last call, and use the same matrices
; COMMON BLOCKS:
;	deriv_dy_saved, holds the coefficient matrices for the grid
; HISTORY:
;	1-apr-2003, jmm, jimm@ssl.berkeley.edu, from deriv.pro
;       24-may-2005, jmm, y01, y02, y12 need not be saved, sy's and
;       ry's can be scalars
;-
Function Deriv_dy, y, f, use_saved_grids = use_saved_grids, _extra = _extra


;Saved grids
  common deriv_dy_saved, cy1, cy2, cy3, sy1, sy2, sy3, ry1, ry2, ry3

;Error handler
  err_dfdy = 0
  catch, err_dfdy
  If(err_dfdy Ne 0) Then Begin
    catch, /cancel
    Print, 'Error'
    help, /last_message, output = theerrormessage
    For j = 0, n_elements(theerrormessage)-1 Do print, theerrormessage[j]
    print, 'Returning -1'
    Return, -1
  Endif
  n = n_elements(y)
  n2 = n-2

  sizf = size(f)
  If(sizf[0] Ne 3) Then message, 'F Array must be 3-d'
  nx = sizf[1] & ny = sizf[2] & nz = sizf[3]

  If(n Lt 3) Then message, 'Parameters must have at least 3 points'
  If(n ne ny) Then message, 'Mismatch between y and f[0,*,0]'

;df/dx=y0*(2x-x1-x2)/(x01*x02)+y1*(2x-x0-x2)/(x10*x12)+y2*(2x-x0-x1)/(x20*x21)
;Where: x01 = x0-x1, x02 = x0-x2, x12 = x1-x2, etc.
  If(Not Keyword_set(use_saved_grids)) Then Begin
    init: type = size(y, /type)    ;If not floating type, ensure floating...
    If((type Ne 4) And (type Ne 5) And (type Ne 6) And (type Ne 9)) Then Begin
      yy = float(y)
      y12 = yy - shift(yy, -1)  ;y1 - y2
      y01 = shift(yy, 1) - yy   ;y0 - y1
      y02 = shift(yy, 1) - shift(yy, -1) ;y0 - y2
    Endif Else Begin            ;Already floating or double
      y12 = y - shift(y, -1)    ;y1 - y2
      y01 = shift(y, 1) - y     ;y0 - y1
      y02 = shift(y, 1) - shift(y, -1) ;y0 - y2
    Endelse
    cy1 = (y12 / (y01*y02))
    cy2 = (1./y12 - 1./y01)
    cy3 = (y01 / (y02 * y12))
    sy1 = (y01[1]+y02[1])/(y01[1]*y02[1])
    sy2 = y02[1]/(y01[1]*y12[1])
    sy3 = (y01[1] / (y02[1] * y12[1]))
    ry1 =  y12[n2]/(y01[n2]*y02[n2])
    ry2 = y02[n2]/(y01[n2]*y12[n2])
    ry3 = (y02[n2]+y12[n2])/(y02[n2]*y12[n2])
;Replicate to size of f
    cy1 = transpose(rebin(cy1, ny, nx, nz, /sample), [1, 0, 2])
    cy2 = transpose(rebin(cy2, ny, nx, nz, /sample), [1, 0, 2])
    cy3 = transpose(rebin(cy3, ny, nx, nz, /sample), [1, 0, 2])
  Endif Else Begin
    If(n_elements(cy1) Eq 0) Then Goto, init
    If(n_elements(cy1[0, *, 0]) Ne ny) Then Goto, init
  Endelse
;Do the derivative
  d = f & d[*] = 0.0
  d = shift(f, 0, 1, 0) * cy1 + $ ;Middle points
    f * cy2 - $
    shift(f, 0, -1, 0) * cy3
; Formulae for the first and last points:
  d[*, 0, *] = f[*, 0, *] * sy1 - $ ;First point
      f[*, 1, *] * sy2 + $
      f[*, 2, *] * sy3
  d[*, n-1, *] = -f[*, n-3, *] * ry1 + $ ;Last point
    f[*, n-2, *] * ry2 - $
    f[*, n-1, *] * ry3
  
  Return, d
End
