;+
; NAME:
;	Deriv_dx
; PURPOSE:
;       Partial Derivative of a 3-d array with respect to the first
;       (x) index
; CALLING SEQUENCE:
;	Dfdx = Deriv_dx(x, f, use_saved_grids=use_saved_grids)
; INPUTS:
;	X:  Variable to differentiate with respect to
;	f:  Function to be differentiated, must be 3-d with the first
;	index having the same number of elements as X
; OUTPUTS:
;	dfdx= the derivative df/dx at each point
; KEYWORDS:
;       use_saved_grids, if set then assume that the x grid hasn't
;       changed since the last call, and use the same matrices
; COMMON BLOCKS:
;	deriv_dx_saved, holds the coefficient matrices for the grid
; HISTORY:
;	1-apr-2003, jmm, jimm@ssl.berkeley.edu, from deriv.pro
;       24-may-2005, jmm, x01, x02, x12 need not be saved, sx's and
;       rx's can be scalars
;-
Function Deriv_dx, x, f, use_saved_grids = use_saved_grids, _extra = _extra


;Saved grids
  common deriv_dx_saved, cx1, cx2, cx3, sx1, sx2, sx3, rx1, rx2, rx3

;Error handler
  err_dfdx = 0
  catch, err_dfdx
  If(err_dfdx Ne 0) Then Begin
    catch, /cancel
    Print, 'Error'
    help, /last_message, output = theerrormessage
    For j = 0, n_elements(theerrormessage)-1 Do print, theerrormessage[j]
    print, 'Returning -1'
    Return, -1
  Endif
  n = n_elements(x)
  n2 = n-2

  sizf = size(f)
  If(sizf[0] Ne 3) Then message, 'F Array must be 3-d'
  nx = sizf[1] & ny = sizf[2] & nz = sizf[3]

  If(n Lt 3) Then message, 'Parameters must have at least 3 points'
  If(n ne nx) Then message, 'Mismatch between x and f[*,0,0]'

;df/dx=y0*(2x-x1-x2)/(x01*x02)+y1*(2x-x0-x2)/(x10*x12)+y2*(2x-x0-x1)/(x20*x21)
; Where: x01 = x0-x1, x02 = x0-x2, x12 = x1-x2, etc.
  If(Not Keyword_set(use_saved_grids)) Then Begin
    init: type = size(x, /type)    ;If not floating type, ensure floating...
    If((type Ne 4) And (type Ne 5) And (type Ne 6) And (type Ne 9)) Then Begin
      xx = float(x)
      x12 = xx - shift(xx, -1)  ;x1 - x2
      x01 = shift(xx, 1) - xx   ;x0 - x1
      x02 = shift(xx, 1) - shift(xx, -1) ;x0 - x2
    Endif Else Begin            ;Already floating or double
      x12 = x - shift(x, -1)    ;x1 - x2
      x01 = shift(x, 1) - x     ;x0 - x1
      x02 = shift(x, 1) - shift(x, -1) ;x0 - x2
    Endelse
    cx1 = (x12 / (x01*x02))
    cx2 = (1./x12 - 1./x01)
    cx3 = (x01 / (x02 * x12))
    sx1 = (x01[1]+x02[1])/(x01[1]*x02[1])
    sx2 = x02[1]/(x01[1]*x12[1])
    sx3 = x01[1]/(x02[1]*x12[1])
    rx1 = x12[n2]/(x01[n2]*x02[n2])
    rx2 = x02[n2]/(x01[n2]*x12[n2])
    rx3 = (x02[n2]+x12[n2])/(x02[n2]*x12[n2])
;Replicate to size of f
    cx1 = rebin(cx1, nx, ny, nz, /sample)
    cx2 = rebin(cx2, nx, ny, nz, /sample)
    cx3 = rebin(cx3, nx, ny, nz, /sample)
  Endif Else Begin
    If(n_elements(cx1) Eq 0) Then Goto, init
    If(n_elements(cx1[*, 0, 0]) Ne nx) Then Goto, init
  Endelse
;Do the derivative
  d = f & d[*] = 0.0
  d = shift(f, 1, 0, 0) * cx1 + $ ;Middle points
    f * cx2 - $
    shift(f, -1, 0, 0) * cx3
; Formulae for the first and last points:
  d[0, *, *] = f[0, *, *] * sx1 - $ ;First point
      f[1, *, *] * sx2 + $
      f[2, *, *] * sx3
  d[n-1, *, *] = -f[n-3, *, *] * rx1 + $ ;Last point
    f[n-2, *, *] * rx2 - $
    f[n-1, *, *] * rx3
  
  Return, d
End
