;+
;NAME:
; calc_lin_fff
;PURPOSE:
; Calculation to determine linear FFF from the field on the lower
; boundary. Note that this only works for cartesian coordinates
;CALLING SEQUENCE:
; calc_lin_fff, x, y, z, dx, dy
;INPUT:
; x, y, z = the spatial grid
; dx, dy = dx and dy for the x, y grids
;In common block bfields:
; pbx, pby, pbz = the poential of linear FFF field solution
; bx, by, bz = the reconstructed b field
;OUTPUT:
; In common block bfields:
; pbx, pby, pbz = the poential field solution
;HISTORY:
; 2-oct-2000, jmm, jimm@ssl.berkeley.edu
; Added non-symmetric x, y, 17-feb-2005, jmm
; x, y, z are in the common block, 7-sep-2005, jmm
; 13-sep-2005, jmm, speeding up, made sigle precision to save memory,
; removed loop over z
;-
Pro Calc_lin_fff, dx, dy, alpha, shrink_factor = shrink_factor, $
                  _extra = _extra
   
  Common Bfields, bx, by, bz, pbx, pby, pbz, x, y, z, $
    rsize, tsize, vsize, rsize1, tsize1, vsize1

  If(keyword_set(shrink_factor)) Then Begin
;Probably you can only use this with uniform grids...
    rx = long(rsize/shrink_factor)
    tx = long(tsize/shrink_factor)
    vx = long(vsize/shrink_factor)
    If(rx mod 2 Eq 0) Then rx = rx+1l ;odd numbers
    If(tx mod 2 Eq 0) Then tx = tx+1l
    If(vx mod 2 Eq 0) Then vx = vx+1l
    xx = congrid(float(x), rx, /interp)
    yy = congrid(float(y), tx, /interp)
    zz = congrid(float(z), vx, /interp)
    dxx = [xx[1:*]-xx] & dxx = [dxx, dxx[rx-2]]
    dyy = [yy[1:*]-yy] & dyy = [dyy, dyy[tx-2]]
    bzx = congrid(bz[*, *, 0], rx, tx)
    calc_lin_fff_a, bzx, xx, yy, zz, dxx, dyy, alpha, pxx, pyy, pzz
    pbx = congrid(pxx, rsize, tsize, vsize)
    pby = congrid(pyy, rsize, tsize, vsize)
    pbz = congrid(pzz, rsize, tsize, vsize)
    Return
  Endif
  If(alpha Ne 0.0) Then Begin
    message, /info, 'Linear FFF implemented, alpha='+$
      strcompress(string(alpha))
  Endif Else message, /info, 'Potential Field implemented'
;Simpson's rule coefficients, if dx and dy are constant
;Also it helps if there is an odd number of components
  coeffxp = fltarr(rsize, tsize)
  coeffyp = coeffxp
;  If(max(dx) Eq min(dx) And Max(dy) Eq min(dy)) Then Begin
;    coeffx = dx
;    coeffy = dy
;    For i = 0, rsize-1 Do Begin
;      If(i Eq 0 Or i Eq rsize-1) Then coeffx[i] = dx[i]/3.0 Else Begin
;        If((i Mod 2) Eq 0) Then coeffx[i] = 4.0*dx[i]/3.0 $
;        Else coeffx[i] = 2.0*dx[i]/3.0
;      Endelse
;    Endfor
;    For j = 0, tsize-1 Do Begin
;      If(j Eq 0 Or j Eq tsize-1) Then coeffy[j] = dy[j]/3.0 Else Begin
;        If((j Mod 2) Eq 0) Then coeffy[j] = 4.0*dy[j]/3.0 $
;        Else coeffy[j] = 2.0*dy[j]/3.0
;      Endelse
;    Endfor
;  Endif Else Begin
;Trapezoid rule coefficients
    coeffx = 0.5*[dx[0], dx[0:rsize-3]+dx[1:rsize-2], dx[rsize-2]]
    coeffy = 0.5*[dy[0], dy[0:tsize-3]+dy[1:tsize-2], dy[tsize-2]]
;  Endelse
;Arrays for integration, need to be 3d to simplify loop structure
  coeffxp = rebin(coeffx, rsize, tsize, vsize)
  coeffyp = transpose(rebin(coeffy, tsize, vsize, rsize), [2, 0, 1])
  twopie = 2.0*!Pi
  xp = rebin(float(x), rsize, tsize, vsize)
  yp = transpose(rebin(float(y), tsize, vsize, rsize), [2, 0, 1])
  zp = transpose(rebin(float(z), vsize, rsize, tsize), [1, 2, 0])
  cos_az = cos(alpha*zp)
  sin_az = sin(alpha*zp)
  bzp = rebin(bz[*, *, 0], rsize, tsize, vsize)
  For j = 0, tsize1 Do  For i = 0, rsize1 Do Begin
    bigR = sqrt((float(x[i])-xp)^2+(float(y[j])-yp)^2)
    bigR = rebin(temporary(bigR), rsize, tsize, vsize)
    r = sqrt(bigR^2+zp^2)
    cos_ar = cos(alpha*r)
    sin_ar = sin(alpha*r)
;to avoid messages
    xxx = where(bigr Eq 0)
    If(xxx[0] Ne -1) Then bigr[xxx] = 1.0e20
    yyy = where(r Eq 0)
    If(yyy[0] Ne -1) Then r[yyy] = 1.0e20
    gamma = zp*cos_ar/(bigR*r)-cos_az/bigR
    dgammadz = cos_ar*(1.0/(bigr*r)-zp^2/(bigr*r^3))- $
      alpha*zp^2*sin_ar/(bigr*r^2)+alpha*sin_az/bigr
    gx = bzp*((x[i]-xp)*dgammadz/bigr+alpha*gamma*(y[j]-yp)/bigr)
    gy = bzp*((y[j]-yp)*dgammadz/bigr-alpha*gamma*(x[i]-xp)/bigr)
    gz = bzp*(zp*cos_ar/r^3+alpha*zp*sin_ar/r^2)
;explicitly set to zero the i, jth points
    gx[i, j, *] = 0.0
    gy[i, j, *] = 0.0
    gz[i, j, *] = 0.0
;gx, gy and gz are the integrands, integrate..
    pbx[i, j, *] = total(total(coeffxp*coeffyp*gx, 1), 1)/twopie
    pby[i, j, *] = total(total(coeffxp*coeffyp*gy, 1), 1)/twopie
    pbz[i, j, *] = total(total(coeffxp*coeffyp*gz, 1), 1)/twopie
;    If(i eq 9 and j eq 3) Then Begin
;      print, x[i], y[j]
;      print, xp[0, 0, 1], yp[0, 0, 1]
;      print, bigr[0, 0, 1], r[0, 0, 1], cos_ar[0, 0, 1], sin_ar[0, 0, 1]
;      print, gamma[0, 0, 1], dgammadz[0, 0, 1]
;      print, bzp[0, 0, 1]
;      print, gx[0, 0, 1], gy[0, 0, 1], gz[0, 0, 1]
;      print, coeffxp[0, 0, 1], coeffyp[0, 0, 1]
;      print, pbx[i, j, 1], pby[i, j, 1], pbz[i, j, 1]
;    Endif
  Endfor
  pbz[*, *, 0] = bz[*, *, 0]    ;otherwise it's 0..
  delvarx, coeffxp, coeffyp, xp, yp, zp
  delvarx, cos_az, sin_az, cos_ar, sin_ar
  delvarx, gx, gy, gz, gamma, dgammadz
  delvarx, r, bigr, bzp
  Return
End
