;+
;^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
;
;  pfss_mag_create.pro - This procedure resamples a magnetic map of the full
;                        solar surface (such as a synoptic map) onto either a
;                        Legendre grid or an equally spaced lat-lon grid.  The
;                        Legendre grid is compatible with the PFSS field
;                        extrapolation software included in this package.
;
;  usage: pfss_mag_create,magout,magtype,nlat,gridtype,file=file,/quiet
;         where magout = output magnetogram image in (lon,lat) format
;               magtype = 0=IDL save file containing flux concentrations
;                         1=Wilcox text file (see below)
;                         2=GONG 360x180 magnetogram in FITS format
;                         3= ? (others can be added in the future...)
;               nlat = number of gridpoints in latitude (default=48)
;               gridtype = 1=regularly spaced grid in lat and lon
;                          2=Legendre grid (default)
;               file = filename of input magnetic data
;               quiet = set if minimal screen output is desired
;
;  notes: -This procedure merely resamples the input data.  It does *NOT*
;          convert from line-of-sight magnetic fields typically measured in
;          synoptic maps to Br (as would be needed to do a PFSS
;          extrapolation).
;
;         -The total (unsigned) flux of the input magnetogram, proportional to
;              mean(abs(data))*4*!dpi*R^2,
;          should about equal the total flux of the resampled magnetogram,
;              total(total(abs(magout),1)/nlon*weights)*2*!dpi*R^2.
;          The percentage difference between the two (sometimes as large as
;          5%) arises due to the fact that bilinear interpolation is used to
;          resample from a coarse grid, and from the fact that the gridpoints
;          near the poles are populated with the flux density of the closest
;          nearest gridpoint in the input synoptic map. 
;
;         -One can display the resulting magnetogram in orthographic 
;          projection using 
;            @pfss_data_block  ;  gets lat and lon arrays
;            scim,mag,m=4,ortho=[270,0],olon=lon,olat=lat
;
;         -magtype=0: IDL save file contains the following four arrays:
;                       nflux = number of flux concentrations
;                       fluxs = array of fluxes in units of Mx
;                       phis = array of longitudes in radians
;                       thetas = array of colatitudes in radians
;                     To get Mx/cm^2 at each point, multiply by
;                       nlat*nlon/(4*!dpi*rsun^2) where rsun=6.959d18 (cm)
;         -magtype=1: Wilcox text files are available at
;                     http://wso.stanford.edu/synoptic.html, with the
;                     input data in units of microTesla.
;         -magtype=2: GONG synoptic maps not yet widely available, but
;                     for now, ask a GONGster (http://gong.nso.edu)
;
;  M.DeRosa - 30 Jan 2002 - converted from earlier script
;              5 Mar 2002 - now successfully reads in Wilcox magnetogram 
;                           tables without proper spacing between columns
;             27 Jun 2002 - added quiet keyword
;              8 Nov 2002 - added capability to resample onto rectangular
;                           grid, added gridtype parameter
;             12 May 2003 - converted common block to PFSS package format
;             21 May 2003 - added magtype = 0, based on output from Karel's
;                           flux-transport model
;              9 Mar 2005 - now parses WSO text files that do not start at 0
;                           degrees longitude
;              9 Mar 2005 - fixed registration error when parsing WSO synoptic
;                           maps (longitudes formerly ranged from 0 to 355
;                           when they should have ranged from 5 to 360)
;  G.Petrie - 14 Jul 2006 - added magtype=2 (GONG synoptic maps)
;              
;^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
;-

pro pfss_mag_create,magout,magtype,nlat0,gridtype,file=file,quiet=quiet

;  print usage message
if n_params() eq 0 then begin
  print,'  pfss_mag_create,magout,magtype,nlat,gridtype,file=file,/quiet'
  return
endif

;  include common block
@pfss_data_block

;  set gridsize
if n_elements(nlat0) eq 0 then nlat=48 else nlat=round(nlat0(0))
nlon=nlat*2

;  set up grid
if n_elements(gridtype) gt 0 then gridtype=gridtype(0) else gridtype=2
case gridtype of
  1: begin  ;  uniformly spaced grid
    lat=linrange(nlat+1,-90,90)
    lat=(lat(0:nlat-1)+lat(1:nlat))/2
    theta=(90-lat)*!dpi/180
    weights=sin(theta)
    weights(0)=.5
    weights(nlat-1)=0.5
    end
  else: begin  ;  Legendre grid
    gaussquad_legendre,nlat,cth,weights
    theta=acos(cth)  ;  radians
    lat=90-theta*180/!dpi
    end
endcase
lon=(linrange(nlon+1,0,360))(0:nlon-1)
phi=lon*!dtor

case magtype of

  0: begin  ;  IDL save file filled with flux concentrations

    restore,file
    phis=(phis+2*!pi) mod (2*!pi)  ;  puts phis between 0 and 2pi

    ;  create magnetogram by adding in each flux
    mag=fltarr(nlon,nlat) 
    if not keyword_set(quiet) then print, $
      '  pfss_mag_create:  adding '+strcompress(nflux,/r)+' sources'    
    for i=0l,nflux-1 do begin

      if not keyword_set(quiet) then $
        pfss_print_time,'  adding sources: ',i+1,nflux,tst,slen

      ;  find out where this source is on our grid
      thc=get_interpolation_index(lat,90-!radeg*thetas(i))
      thc1=fix(thc)  &  thc2=(thc1+1)<(nlat-1)
      phc=get_interpolation_index([phi,!dpi*2],phis(i))
      phc1=fix(phc)  &  phc2=(phc1+1) mod nlon

      ;  add flux to grid, and divide by areal factor
      bco=(thc-thc1)
      aco=(phc-phc1)
      mag(phc1,thc1)=mag(phc1,thc1)+(1-aco)*(1-bco)*fluxs(i)/weights(thc1)
      mag(phc2,thc1)=mag(phc2,thc1)+aco*(1-bco)*fluxs(i)/weights(thc1)
      mag(phc1,thc2)=mag(phc1,thc2)+(1-aco)*bco*fluxs(i)/weights(thc2)
      mag(phc2,thc2)=mag(phc2,thc2)+aco*bco*fluxs(i)/weights(thc2)

    endfor
    magout=mag*mean(weights)  ;  normalization

    end

  1: begin  ;  Wilcox magnetogram

    ;  idiot check
    if (n_elements(file) eq 0) then begin
      print,'  pfss_mag_create: filename must be specified'
      return
    endif

    ;  parse file
    openr,lun,file,/g
    fs=fstat(lun)
    tablebyte=bytarr(fs.size)
    readu,lun,tablebyte
    table=string(tablebyte)
    ix=strpos(table,'CT')
    posix=strpos(table,'CT',ix+1)
    repeat begin
      ix=[ix,posix]
      posix=strpos(table,'CT',posix+1)
    endrep until posix eq -1
    longitudes=round(float(strmid(table,ix+7,3)))
    ix=ix+18
    point_lun,lun,0
    data=fltarr(72,30)  ;  assumes 72 lon bins, 30 slat bins
    for i=0,71 do begin
      point_lun,lun,ix(i)
      buff1=fltarr(6)
      readf,lun,buff1,f='(6f9.3)'
      buff2=fltarr(8)
      readf,lun,buff2,f='(8f9.3)'
      buff3=fltarr(8)
      readf,lun,buff3,f='(8f9.3)'
      buff4=fltarr(8)
      readf,lun,buff4,f='(8f9.3)'
      data(i,*)=[buff1,buff2,buff3,buff4]
    endfor
    free_lun,lun
    data=shift(data,-(where(longitudes eq 360))(0),0)  ;  align longitudes to 0
    data=reverse(data,1)  ;  make longitude increasing
    data=shift(data,1,0)  ;  make 360 first longitude instead of 5
    data=reverse(data,2)  ;  make latitude (instead of colatitude) increasing

    ;  remap onto our grid
    dlatix=asin(linrange(30,14.5,-14.5)/15)*180/!dpi  ;  30 slat bins
    dlonix=linrange(72,5,360)  ;  72 longitude bins
    dlatinterp=get_interpolation_index(reverse(dlatix),lat)
    dloninterp=get_interpolation_index(dlonix,lon)
    magout=interpolate(data,dloninterp,dlatinterp,/grid)

    ;  no need to take into account unequal-sized pixels of Legendre grid,
    ;  since both input and output data are in terms of flux densities
    ;  (e.g. Tesla or Gauss) instead of flux (e.g. Weber or Maxwell)

    end

  2: begin  ;  (GONG) 360x180 magnetogram in FITS format

    ;  idiot check
    if (n_elements(file) eq 0) then begin
      print,'  pfss_mag_create: filename must be specified'
      return
    endif

    ;  read fits file
    mreadfits,file,hdr,data
  
    ;  remap onto our grid
    dlatix=asin(linrange(180,89.5,-89.5)/90.0)*180/!dpi  ;  180 slat bins
    dlonix=linrange(360,1,360)  ;  360 longitude bins

    dlatinterp=get_interpolation_index(reverse(dlatix),lat)
    dloninterp=get_interpolation_index(dlonix,lon)
    magout=interpolate(data,dloninterp,dlatinterp,/grid)

    end

  else: begin
    print,'  pfss_mag_create: invalid magtype'
    return
    end

endcase

if not keyword_set(quiet) then print,'  pfss_mag_create:  magnetogram created'

lat0=lat
lon0=lon

end

