;+
;^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
;
;  pfss_potl_field.pro - This procedure computes the potential magnetic field
;                        B(r,theta,phi) given the potential Phi(l,m,r)
;
;  usage: pfss_potl_field,rtop,rgrid,lmax=lmax,/trunc,potl=potl,/quiet
;         where rtop=radius of uppermost gridpoint
;               rgrid=sets radial gridpoint spacing:
;                      1 = equally spaced (default)
;                      2 = grid spacing varies with r^2
;               lmax=if set, only use lmax spherical harmonics in constructing
;                    potential
;               trunc=set to use fewer spherical harmonics when
;                     reconstructing B are you get farther out in radius
;               potl=contains potl if desired, but what you pass
;                    to this routine must not be undefined in order
;                    for the field potential to be computed
;               quiet = set for minimal screen output
;
;         and in the common block we have:
;               phiat=on input, (l,m) array of dcomplex coeffs, 
;                     corresponding to r^l eigenfunction
;               phibt=on input, (l,m) array of dcomplex coeffs, 
;                     corresponding to 1/r^(l+1) eigenfunction
;               (br,bth,bph)=in output, (r,theta,phi) components of B-field
;
;
;  M.DeRosa - 30 Jan 2002 - converted from earlier script
;              8 Feb 2002 - added lmax keyword
;              2 Jul 2002 - added quiet keyword
;             30 Apr 2003 - now utilizes memory more efficiently, based on a
;                           suggestion from Bart De Pontieu
;             12 May 2003 - converted common block to PFSS package format
;
;^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
;-

pro pfss_potl_field,rtop,rgrid,lmax=lmax,trunc=trunc,potl=potl,quiet=quiet

;  print usage message
if n_params() eq 0 then begin
  print,'  pfss_potl_field,rtop,rgrid,lmax=lmax,/trunc,potl=potl,/quiet'
  return
endif

;  include common block
@pfss_data_block

;  preliminaries
if keyword_set(lmax) then lmax=lmax<nlat else lmax=nlat
cth=cos(theta)

;  get l and m index arrays of transform
lix=lindgen(lmax+1)
mix=lix
larr=lix#replicate(1,lmax+1)
marr=replicate(1,lmax+1)#mix
wh=where(marr gt larr)
larr(wh)=0  &  marr(wh)=0

;  get radial grid
dr0=(!dpi/nlat)  ;  r grid spacing at r=1, make it half avg lat grid spacing
rra=[1d0,double(rtop(0))]  ;  range of r
case rgrid of
  2: begin  ;  radial gridpoint separation is proportional to r^2
    rix=[rra(0)]
    lastr=rra(0)
    repeat begin 
      nextr=lastr+dr0*(lastr/rra(0))^2
      rix=[rix,nextr]
      lastr=nextr
    endrep until nextr ge rra(1)
    rix2=rix/((max(rix)-rra(0))/(rra(1)-rra(0)))
    rix=rix2+(rra(0)-rix2(0))
    nr=n_elements(rix)
    end
  else: begin  ;  radial gridpoints uniformly spaced
    nr=round((rra(1)-rra(0))/dr0)
    rix=linrange(nr,rra(0),rra(1))
    end
endcase
if not keyword_set(quiet) then print,'  pfss_potl_field: nr = '+$
  strcompress(nr,/r)

;  set up planar sin(theta) array
stharr=replicate(1,nlon)#sqrt(1-cth*cth)

;  compute lmax for each radius
lmaxarr=lonarr(nr,/noz)
if keyword_set(trunc) then begin  ;  include fewer l modes as you get higher up
  lmaxarr(0)=nlat<lmax
  for i=1,nr-1 do begin
    wh=where(rix(i)^lindgen(nlat+1) gt 1e6,nwh)
    if nwh eq 0 then lmaxarr(i)=lmax else lmaxarr(i)=(wh(0)<lmax)
  endfor
endif else lmaxarr(*)=nlat  ;  otherwise do nlat transforms for all radii

;  compute Br in (r,l,m)-space
bt=make_array(dim=[lmax+1,lmax+1,nr],/noz,/dcomplex)
for i=0,nr-1 do $
  bt(*,*,i)= phiat*larr*rix(i)^(larr-1) - phibt*(larr+1)*rix(i)^(-larr-2)

;  ...and then transform to (r,theta,phi)-space
br=make_array(dim=[nlon,nlat,nr],/float,/noz)
for i=0,nr-1 do begin
  if not keyword_set(quiet) then $
    pfss_print_time,'  pfss_potl_field: computing Br:  ',i+1,nr,tst,slen,/elap
  br(*,*,i)=inv_spherical_transform(bt(*,*,i),cth,lmax=lmaxarr(i))
endfor

;  compute sin(theta) * Bth in (r,l,m)-space...
factor=sqrt(double(larr^2-marr^2)/double(4*larr^2-1))
for i=0,nr-1 do begin
  bt(*,*,i)=(larr-1)*factor* $
    (shift(phiat,1,0)*rix(i)^(larr-2) + shift(phibt,1,0)*rix(i)^(-larr-1)) $
    - (larr+2)*shift(factor,-1,0)* $
    (shift(phiat,-1,0)*rix(i)^larr + shift(phibt,-1,0)*rix(i)^(-larr-3))
  bt(0,0,i)=-2*factor(1,0)*(phiat(1,0) + phibt(1,0)*rix(i)^(-3))
  bt(lmax,*,i)=(lmax-1)*factor(lmax,*)* $
    (phiat(lmax-1,*)*rix(i)^(lmax-2) + phibt(lmax-1,*)*rix(i)^(-lmax-1))
endfor

;  ...and then compute Bth in (r,theta,phi)-space
bth=make_array(dim=[nlon,nlat,nr],/float,/noz)
for i=0,nr-1 do begin
  if not keyword_set(quiet) then $
    pfss_print_time,'  pfss_potl_field: computing Bth:  ',i+1,nr,tst,slen,/elap
  bth(*,*,i)=inv_spherical_transform(bt(*,*,i),cth,lmax=lmaxarr(i))/stharr
endfor

;  compute sin(theta) * Bph in (r,l,m)-space...
for i=0,nr-1 do bt(*,*,i)=complex(0,1)*marr* $
  (phiat*rix(i)^(larr-1) + phibt*rix(i)^(-larr-2))

;  ...and then compute Bph in (r,theta,phi)-space
bph=make_array(dim=[nlon,nlat,nr],/float,/noz)
for i=0,nr-1 do begin
  if not keyword_set(quiet) then $
    pfss_print_time,'  pfss_potl_field: computing Bph:  ',i+1,nr,tst,slen,/elap
  bph(*,*,i)=inv_spherical_transform(bt(*,*,i),cth,lmax=lmaxarr(i))/stharr
endfor

;  now transform the field potential to (r,theta,phi)-space
if n_elements(potl) gt 0 then begin
  potl=make_array(dim=[nlon,nlat,nr],/float,/noz)
  for i=0,nr-1 do begin
    if not keyword_set(quiet) then $
      pfss_print_time,'  pfss_potl_field: computing the field potential:  ',$
        i+1,nr,tst,slen,/elap
      potl(*,*,i)=inv_spherical_transform(phibt*rix(i)^(-larr-1)+ $
        phiat*rix(i)^larr,cth,lmax=lmax)
  endfor
endif

end      
